/*
 * routines to implement a "sync timer" which is a timer that starts ticking
 * only after a packet send completes.
 * The goal is to make this look like a single operation to the caller which 
 * can be safely cancelled anywhere along the process.
 */

#include "libfma.h"
#include "lf_scheduler.h"

#include "fma.h"
#include "fma_sync_timer.h"
#include "fma_myri.h"

/*
 * local prototypes
 */
static void fma_sync_timer_start(void *vstp);
static void fma_sync_timer_popped(void *vstp);

/*
 * Send a packet, and then start a timer once the send completes
 */
struct fma_sync_timer *
fma_myri_raw_send_with_timer(
  int nic_id,
  int port,
  void *route,
  int route_len,
  void *txbuf,
  int txlen,
  void (*callback)(void *),
  void *arg,
  int timeout)
{
  struct fma_sync_timer *stp;
  int rc;

  /* Allocate and fill in sync timer struct */
  LF_CALLOC(stp, struct fma_sync_timer, 1);
  stp->st_callback = callback;
  stp->st_arg = arg;
  stp->st_timeout = timeout;

  /* Send a packet with a callback */
  rc = fma_myri_raw_send(nic_id, port, route, route_len, txbuf, txlen,
			 fma_sync_timer_start, stp);
  if (rc == -1) {
    LF_ERROR(("Error performing send"));
  }
  return stp;

 except:
  LF_FREE(stp);
  fma_perror_exit(1);
  return NULL;
}

/*
 * Send a sync packet, and then start a timer once the send completes
 */
struct fma_sync_timer *
fma_sync_timer_setup(
  int nic_id,
  int port,
  void (*callback)(void *),
  void *arg,
  int timeout)
{
  struct fma_sync_timer *stp;
  int rc;

  /* Allocate and fill in sync timer struct */
  LF_CALLOC(stp, struct fma_sync_timer, 1);
  stp->st_callback = callback;
  stp->st_arg = arg;
  stp->st_timeout = timeout;

  /* Send a packet with a callback */
  rc = fma_myri_sync_callback(nic_id, port, fma_sync_timer_start, stp);
  if (rc == -1) {
    LF_ERROR(("Error setting up sync callback"));
  }
  return stp;

 except:
  LF_FREE(stp);
  fma_perror_exit(1);
  return NULL;
}

/*
 * Cancel this sync timer.  It either is referenced in a callback
 * or else has a timer pending.
 */
void
fma_sync_timer_cancel(
  struct fma_sync_timer *stp)
{
  /* If not started yet, mark it "cancelled" */
  if (stp->st_timer == NULL) {
    stp->st_cancelled = TRUE;

  /* timer is running, clear the timer and free this struct */
  } else {
    lf_remove_event(stp->st_timer);
    LF_FREE(stp);
  }
}

/*
 * Packet send has completed, start the timer unless we have been cancelled
 */
static void
fma_sync_timer_start(
  void *vstp)
{
  struct fma_sync_timer *stp;

  stp = vstp;

  /* If cancelled, just free the struct and disappear */
  if (stp->st_cancelled) {
    LF_FREE(stp);

  /* not cancelled, so start the timer with user supplied callback info */
  } else {
    stp->st_timer = lf_schedule_event(fma_sync_timer_popped, stp,
				      stp->st_timeout);
    if (stp->st_timer == NULL) {
      LF_ERROR(("Error starting timer"));
    }
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * If we got here, we were not cancelled, so make the user's callback
 */
static void
fma_sync_timer_popped(
  void *vstp)
{
  struct fma_sync_timer *stp;

  stp = vstp;

  /* make the callback */
  stp->st_callback(stp->st_arg);

  /* all done!! */
  LF_FREE(stp);
}
